const path = require('path')
const fs = require('fs-extra')
const { EVENTS, PSKEYS } = require('./consts')
const Headset = require('./HeadsetSingleton')
const pskey_convert = require('./PskeyConverter')

const confDir = process.env.APPDATA || (process.platform == 'darwin' ? process.env.HOME + '/Library/Preferences' : process.env.HOME + "/.local/share")

const presetsFile = path.join(confDir ,'cardo-presets.json')

class Presets {

  constructor(ipc) {
    this.ipc = ipc

    //subscribe to events
    this.ipc.on(EVENTS.CREATE_PRESET,(event, arg) => {
      let preset = this.createPreset(arg.name)
      event.sender.send(`${EVENTS.CREATE_PRESET}-response`,preset)
    })

    this.ipc.on(EVENTS.GET_PRESETS,(event, arg) => {
      let result = this.getAll()
      event.sender.send(`${EVENTS.GET_PRESETS}-response`,result)
    })

    this.ipc.on(EVENTS.DELETE_PRESET,(event, arg) => {
      this.deletePreset(arg.name)
      event.sender.send(`${EVENTS.DELETE_PRESET}-response`,true)
    })

    this.ipc.on(EVENTS.SET_PRESET_PSKEY, (event, arg) => {
      let result = this.setPskey(arg.preset, arg.key, arg.value)
      event.sender.send(`${EVENTS.SET_PRESET_PSKEY}-response`,result)
    })

    this.ipc.on(EVENTS.BATCH_PRESET, (event, arg) => {
      const headset = Headset.getInstance()
      var preset = this.getPreset(arg)
      let convertedKeys = {}
      for (let key in preset.settings) {
        convertedKeys[key] =  pskey_convert.convertToPSKeyValue(key, preset.settings[key])
      }

      headset.setPskeys(convertedKeys)
      event.sender.send(`${EVENTS.BATCH_PRESET}-response`, true)
    })
  }

  getPreset(name) {
    return this.getAll().find(preset => preset.name === name)
  }

  setPskey(presetName, key, value) {
    console.log('Setting ' + key + " to: " + value)
    var presets = this.getAll()

    var preset = presets.find(preset => preset.name === presetName)

    if(!preset) return false

    preset.settings[key] = value

    this.savePresets(presets)
    return true
  }

  createPreset(name) {

    const preset = {
      name: name,
      settings: {
        [PSKEYS.language]: "english",
        [PSKEYS.batteryType]: "Li-Ion",
        [PSKEYS.VAD]: "medium",
        [PSKEYS.micSensitivity]: "medium",
        [PSKEYS.FMBand]: "worldwide",
        [PSKEYS.enableFM]: true,
        [PSKEYS.ASR]: true,
        [PSKEYS.RDS]: true,
        [PSKEYS.parallelAudioMixing]: true,
        [PSKEYS.twoWayRadioAudioMixing]: true,
        [PSKEYS.twoWayRadioSharing]: false,
        [PSKEYS.secondaryAudioVolume]: 10,
        [PSKEYS.LD]: false,
        [PSKEYS.LDVolume]: 5,
        "ps_LDAGC": true,
        [PSKEYS.selfHearing]: false,
        [PSKEYS.selfHearingVolume]: 5,
        [PSKEYS.radioPresets]: ["9500","9500","9500","9500","9500","9500"],
        [PSKEYS.enableBT]: true,
        [PSKEYS.intercomChannel]: 1,
        [PSKEYS.speakerGain]: "Level 3",
        [PSKEYS.speakerSwitch]: "Left-Right",
        "ps_VADGain": "Level 4",
        "ps_micGain": "Level 4",
        [PSKEYS.chargeLEDLocation]: "Power Board",
        [PSKEYS.magicWord]: "Hey Cardo" ,
        [PSKEYS.emergencyDialNumber]: "",   
        [PSKEYS.speedDialNumber]: "",   
      }  
    }

    var presets = this.getAll()

    presets.push(preset)
    
    this.savePresets(presets)

    return preset
  }

  savePresets(presets) {
    fs.writeJSONSync(presetsFile,presets,{flag: 'w', encoding: 'utf8'})
  }

  deletePreset(name) {
    var presets = this.getAll()
    presets = presets.filter(preset => preset.name.toLowerCase() !== name.toLowerCase())
    this.savePresets(presets)
  }

  getAll() {
    var presets = []

    if(!fs.existsSync(path.dirname(presetsFile))) {
      fs.mkdirSync(path.dirname(presetsFile))
    }

    if(!fs.existsSync(presetsFile))
    {
      this.savePresets(presets)
      return presets
    }

    presets = fs.readJSONSync(presetsFile,{encoding: 'utf8'})

    return presets
  }
}

module.exports = Presets
